#include "propositional.h"
#include "mode.h"

namespace lp {

	//******* Global Function Definitions ********//

	//******* Vass member definitions *******//

	void Vass::init_model(int model_size)
	{
		pvt = var_table(model_size,pvar());
		va.clear();
		va.reserve(model_size);
	}

	void Vass::init_modes(const Functor& bot, const vector<Mode>& fm) 
	{
		// Create jump table based on modes, make sure same modes refer to same pair
		map<Mode,shared_ptr<pair<int,int>>> m2p; // mode -> pair* mapping

		bit_to_count.reserve(bot.body_size());
		int k = 1;
		for (auto i = bot.body_begin(); i != bot.body_end(); ++i,++k) {
			if (i->is_constant(true_id)) continue;
			const Mode& m = fm[k];
			if (!m) {
				assert( i->is_function(equal2_id,2) ); // currently, we only expect this to happen for variable splitting
				// literal has no mode: set independent "infinite" counter
				shared_ptr<pair<int,int>> p( new pair<int,int>(0,numeric_limits<int>::max()) );
				bit_to_count.push_back(p);
			} else {
				auto at = m2p.find(m);
				if (at == m2p.end()) {
					// new mode, create new shared pointer and update map
					shared_ptr<pair<int,int>> p( new pair<int,int>(0,m.max_occurs()) );
					bit_to_count.push_back(p);
					m2p.insert(make_pair(m,p));
				} else {
					// mode already occured, add same shared pointer
					bit_to_count.push_back(at->second);
				}
			}
		}
	}

	int Vass::insert(const pvar& a) 
	{
		assert( this->check_counter() );

		// Note: if we try to insert an assignment already present, this does not violate max1
		// Hence we check for dupes first, then for max1 violation
		assert( count_if(va.begin(),va.end(),[](const pvar* p){return p->val > 0;}) == ones );
		assert(a.val != 0);
		const int idx = std::abs(a.val)-1;
		pvar& var = pvt[idx];
		//std::cerr << "Vass::insert(" << a.val << "), pvt[" << idx << "] = " << pvt[idx].val << "\n";
		if (var.val != 0) {
			if (var.val == a.val) return 1; // already present
			else return 0; // contradition
		}
		// Check max1 and recall restriction
		if (a.val > 0) {
			// check ma1
			//std::cerr << "ones == m1: " << ones << " == " << m1 << "\n";
			if (ones == m1) return -2; // clause length exceeded
			// check max_occurs
			shared_ptr<pair<int,int>>& sptr = bit_to_count[a.val-1];
			//std::cerr << sptr->first << " >= " << sptr->second << "\n";
			if (sptr->first >= sptr->second) return -1; // max_occurs exceeded
			// ok, update
			++ones;
			++(sptr->first);
		}
		// Ok, add assignment
		if (++ac >= amax) throw max_assignments();
		pvt[idx] = a;
		va.push_back(&pvt[idx]);
		assert(ones <= m1 && check_counter() );
		assert( count_if(va.begin(),va.end(),[](const pvar* p){return p->val > 0;}) == ones );
		return 2; // success
	}

	void Vass::pop() 
	{
		const int l = va.back()->val;
		if (va.back()->level < 0) throw unmodifiable();
		if (l > 0) {
			--ones;
			shared_ptr<pair<int,int>>& sptr = bit_to_count[l-1];
			--(sptr->first);
		}
		auto& pref = pvt[std::abs(l)-1];
		pref.val = 0;
		pref.ante = nullptr;
		pref.both_ways = false;
		pref.level = -1;
		pref.sublevel = -1;
		va.pop_back();
	}

	int Vass::flip_back()
	{
		assert( count_if(va.begin(),va.end(),[](const pvar* p){return p->val > 0;}) == ones );
		auto& ref = va.back();
		if (ref->level < 0) return 0;
		if (ref->both_ways) return 0;
		if (ref->val < 0) {
			if (filled1()) return 0; // cannot flip it
			shared_ptr<pair<int,int>>& sptr = bit_to_count[-ref->val - 1];
			if (sptr->first >= sptr->second) return 0; // mode recall violation
			++(sptr->first);
			++ones; // flipping will give another one
		} else {
			assert(ones > 0);
			--ones; // va[i] is positive, flipping will give one less 1
			shared_ptr<pair<int,int>>& sptr = bit_to_count[ref->val-1];
			--(sptr->first);
		}
		// flip assignment
		if (++ac >= amax) throw max_assignments();
		assert(ref->sublevel == 0 && ref->val != 0 && ref->ante == 0);
		ref->val = -ref->val;
		ref->ante = nullptr;
		ref->both_ways = true;
		// level is kept
		assert( count_if(va.begin(),va.end(),[](const pvar* p){return p->val > 0;}) == ones );
		assert( check_counter() );
		return 1;
	}

	void Vass::print(ostream& os) const 
	{
		static const char* sep = "*";
		os << "(";
		auto v = va.begin();
		if (v != va.end()) {
			os << (*v)->val << sep << (*v)->level;
			for (++v; v != va.end(); ++v) os << "," << (*v)->val << sep << (*v)->level;
		}
		os << ")";
	}


	//******* PClause member definitions *******//

	void PClause::print(ostream& os) const 
	{
		os << "[";
		auto i = lits.begin();
		if (i != lits.end()) {
			os << *i;
			for (++i; i != lits.end(); ++i) os << "," << *i;
			if (watch.first) {
				os << "|";
				os << watch.first;
				if (watch.second) os << "," << watch.second;
			}
		}
		os << "]";
	}



	// ============== ClauseDB members ================ //
	/* Constraints available:
		1. Head outputs are used ("return constraints")
		2. Inputs to body literals are instantiated ("chaining constraints")
		3. Head inputs are used ("argument constraints")
		4. Ouputs to body literals are used ("functional predicates")
		5. Head-connectedness ("weak chaining constraints")
	*/

	void ClauseDB::init_modes(
		const Functor& f, 
		const vector<Mode>& fm,
		Constraints& constr,
		const parameters& params)
	{
		// Compute and store each bit's corresponding modeb and count
		// this is done in the constructor for va: va.init_modes(f,fm);

		// h <- a,b,c
		// X->Y iff X has output var that is in Y
		// If X is active, then it depends on having its input variables instantiated
		// not(X) OR ((A or B) and (C or D)) = -X \/ (C1 /\ C2 /\ ...)
		// = (-X \/ C1) /\ (-X \/ C2) /\ ...

		DEBUG_TRACE(
			cerr << "make_mode_clause:\n";
			cerr << "functor f: " << f << "\n";
			cerr << "modes:\n";
			for_each(fm.begin(),fm.end(),[&](const Mode& m){ cerr << "  " << m << "\n"; });
		);

		// Initialize input and output vectors
		// const int f_body_size = static_cast<int>(f.body_size());

		// Store head inputs and outputs
		const io_map iomap = make_io_map(f,fm);
		const set<id_type>& head_ivars = iomap[0].first;
		const set<id_type>& head_ovars = iomap[0].second;

		// Positions where non splitting equalities begin
		auto lit_begin = f.body_begin();
		int idx_begin = 0;
		// Variable splitting constraints
		// For each splitting equality _=(X,Y), require that it is used by some non-equality literal
		if (params.is_set(parameters::splitvars)) {
			// Calculate where split equalities end
			lit_begin = std::find_if(f.body_begin(),f.body_end(),[&](const Functor& g){ return !g.is_function(equal2_id,2); });
			idx_begin = std::distance(f.body_begin(),lit_begin);

			int ki = 0;
			for (auto i = f.body_begin(); i != lit_begin; ++i, ++ki) {
				assert(i->is_function(equal2_id,2));
				for (int a = 0; a < 2; ++a) { // left and right variable
					assert( i->arg(a)->is_variable() );
					const id_type var = i->arg(a)->id();
					if (head_ivars.find(var) == head_ivars.end() && head_ovars.find(var) == head_ovars.end()) {
						// Not in head, build constraint
						PClause eqcon(-ki-1);
						int kj = idx_begin;
						for (auto j = lit_begin; j != f.body_end(); ++j, ++kj) {
							assert( !j->is_function(equal2_id,2) );
							if (j->is_variable(var) || j->is_variable(var)) eqcon.insert(kj+1);
							// if ((*j)->find(var_right,0)) wc_right.insert(kj+1);
						}
						// Add clause
						this->insert(std::move(eqcon));
						if (this->has_empty()) return;
					}
				}
			}
		}

		// Output Constraints:
		// For a clause to be valid, all -types in head must be given by body literals
		// note: -types in head can be received from same +types in head too
		// note: we do these constraints first as they are able to produce an empty clause
		if (params.is_set(parameters::chain_return)) {
			for (auto v = head_ovars.begin(); v != head_ovars.end(); ++v) {
				// Special case: if -type is also +type in head, we can safely ignore it
				if (head_ivars.find(*v) != head_ivars.end()) continue;
				// Which body literals output this variable?
				PClause lout;
				auto l = f.body_begin();
				int n = 0;
				for ( ; l != lit_begin; ++l, ++n) {
					assert( l->is_function(equal2_id,2) );
					if (l->is_variable(*v) || l->is_variable(*v)) lout.insert(n+1);
				}
				assert( l == lit_begin && n == idx_begin );
				for ( ; l != f.body_end(); ++l, ++n) {
					if (l->is_constant(true_id)) continue;
					const set<id_type>& ovars = iomap[n+1].second;
					if (ovars.find(*v) != ovars.end()) {
						lout.insert(n+1);
					}
				}

				// note: this may insert an empty clause, indicating inconsistency
				// detect inconsistency as early as possible and terminate constraint build
				//std::cerr << "Chained return, adding\n";
				this->insert(std::move(lout));
				if (this->has_empty()) return; // prematurely end if empty clause was added
			}
		}

		// Argument Constraints:
		// All +types in head must be used by body literals (either as input or output)
		// note: +types in head can be received from -types in head too
		const bool param_chain_arguments = params.is_set(parameters::chain_arguments);
		const int param_detect_dependent = params.force_int(parameters::detect_dependent);
		if (param_chain_arguments || param_detect_dependent > 0) {
			std::set<id_type> dvars; // dependent variables
			if (!param_chain_arguments) {
				// Only chain arguments that are dependent
				auto rng = constr.dependent_vars.equal_range(fm.front());
				for (auto i = rng.first; i != rng.second; ++i) {
					const Functor* tmp = f.head()->get(i->second);
					assert(tmp->is_variable());
					//DEBUG_BASIC(std::cout << "Detected dependent variable: " << Functor::get_data(tmp->id()) << "\n");
					dvars.insert(tmp->id());
				}
			}
			for (auto v = head_ivars.begin(); v != head_ivars.end(); ++v) {
				if (!param_chain_arguments) {
					// Only chain dependent variables
					if (dvars.find(*v) == dvars.end()) continue;
				}
				// Special case: if +type is also -type in head, we can safely ignore it
				if (head_ovars.find(*v) != head_ovars.end()) continue;
				// Which body literals takes v as input or output?
				PClause lout;
				auto l = f.body_begin();
				int n = 0;
				for ( ; l != lit_begin; ++l, ++n) {
					assert( l->is_function(equal2_id,2) );
					if ( l->is_variable(*v) || l->is_variable(*v)) lout.insert(n+1);
				}
				assert( l == lit_begin && n == idx_begin );
				for ( ; l != f.body_end(); ++l, ++n) {
					if (l->is_constant(true_id)) continue;
					const set<id_type>& ivars = iomap[n+1].first;
					if (ivars.find(*v) != ivars.end()) {
						lout.insert(n+1);
					} else {
						// Try output variables
						const set<id_type>& ovars = iomap[n+1].second;
						if (ovars.find(*v) != ovars.end()) {
							lout.insert(n+1);    //std::cerr
						}
					}
				}

				// note: this may insert an empty clause, indicating inconsistency
				// detect inconsistency as early as possible and terminate constraint build
			    //std::cerr << "Chained arguments, adding\n";
				this->insert(std::move(lout));
				if (this->has_empty()) return; // prematurely end if empty clause was added
			}
		}

		// Input Constraints: create I/O dependency of literals
		// Each +type in body lit must come from previous -type
		// Note: do NOT skip first literal, as we must check inputs come from head
		if (params.is_set(parameters::chain_io)) {
			int ki = idx_begin;
			for (auto i = lit_begin; i != f.body_end(); ++i, ++ki) {
				//cerr << "Input variables of " << *l << ":\n";
				const set<id_type>& ivars = iomap[ki+1].first;
				for (auto v = ivars.begin(); v != ivars.end(); ++v) {
					// If input variable grabs output from head, there is no restriction
					if (head_ivars.find(*v) != head_ivars.end()) continue;
					// For each input variable, find all possible output literals
					// Note: when using variable splitting, the equality constraints can be
					// used to instantiate a literal too
					PClause lout(-ki-1); // (-ki-1) OR output from X OR output from ...
					auto j = f.body_begin();
					int kj = 0;
					for ( ; j != lit_begin; ++j, ++kj) {
						assert( j->is_function(equal2_id,2) );
						if (j->is_variable(*v) || j->is_variable(*v)) lout.insert(kj+1);
					}
					assert(kj == idx_begin && j == lit_begin);
					for ( ; j != i; ++j, ++kj) { // all previous body literals
						if (j->is_constant(true_id)) continue;
						const set<id_type>& ovars = iomap[kj+1].second;
						if (ovars.find(*v) != ovars.end()) {
							//cerr << n << " ";
							lout.insert(kj+1);
						}
					}
					//cerr << "\n";
					// Add mode restriction clause (is unit clause if no previous input exists)
				    //std::cerr << "Chained IO, adding\n";
					this->insert(std::move(lout));
					if (this->has_empty()) return; // prematurely end if empty clause was added
				}
			}
		}

		// Use head-connected constraints
		if (params.is_set(parameters::weak_chain_io)) {
			// A literal can only be used if at least one variable is instantiated
			int ki = idx_begin;
			for (auto i = lit_begin; i != f.body_end(); ++i, ++ki) {
				const set<id_type>& ivars = iomap[ki+1].first;
				const set<id_type>& ovars = iomap[ki+1].second;
				// Check if a variable is already in head
				if (intersects(ivars,head_ivars) || intersects(ivars,head_ovars) 
					|| intersects(ovars,head_ivars) || intersects(ovars,head_ovars)) {
						continue; // no restrictions for this literal
				}
				// Grab all previous literals that instantiate this one
				PClause wc(-ki-1);
				int kj = idx_begin;
				for (auto j = lit_begin; j != i; ++j, ++kj) {
					const set<id_type>& ivars2 = iomap[kj+1].first;
					const set<id_type>& ovars2 = iomap[kj+1].second;
					if (intersects(ivars,ivars2) || intersects(ivars,ovars2) 
						|| intersects(ovars,ivars2) || intersects(ovars,ovars2)) {
							// This literal may be used
							wc.insert(kj+1);
					}
				}
				// Add clause
				this->insert(std::move(wc));
				if (this->has_empty()) return; // prematurely end if empty clause was added
			}
		}

		// Use purely functional constraints?
		// Each function predicate must have output that is used
		if (params.is_set(parameters::functional)) {
			int ki = idx_begin;
			for (auto i = lit_begin; i != f.body_end(); ++i, ++ki) {
				if (fm[ki+1].props[Mode::functional]) {
					// Check that the output is not in the head
					const set<id_type>& ovars = iomap[ki+1].second;
					if (intersects(ovars,head_ivars) || intersects(ovars,head_ovars)) continue;
					// Require that at least one output is used somewhere else
					PClause fcon(-ki-1);
					// For each output, find all possible input and output variables
					for (auto ov = ovars.begin(); ov != ovars.end(); ++ov) {
						int kj = idx_begin;
						for (auto j = lit_begin; j != f.body_end(); ++j, ++kj) {
							if (i == j) continue; // skip same
							// Check input
							const set<id_type>& ivars2 = iomap[kj+1].first;
							if (ivars2.find(*ov) != ivars2.end()) {
								fcon.insert(kj+1);
							} else {
								// No input, check output
								const set<id_type>& ovars2 = iomap[kj+1].second;
								if (ovars2.find(*ov) != ovars2.end()) {
									fcon.insert(kj+1);
								}
							}
						}
					}
					// Add clause
					this->insert(std::move(fcon));
					if (this->has_empty()) return; // prematurely end if empty clause was added
				}
			}
		} // if functional is set
	}

	bool ClauseDB::satisfied(const Vass& vass) const
	{
		std::cerr << "Satisfied? " << vass << "\n";
		if (has_empty()) return false;
		// Satisfy all unit clauses
		for (auto v = ass.begin(); v != ass.end(); ++v) {
			const pvar& pv = vass.get_var(*v);
			if (*v == -pv.val) {
				std::cerr << "  contradiction: " << pv.val << " against unit [" << *v << "]\n";
				return false;
			}
		}
		// Satisfy all non-unit clauses
		for (auto c = db.begin(); c != db.end(); ++c) {
			if (!c->satisfied(vass)) {
				std::cerr << "  contradiction: clause: " <<  "[" << *c << "]\n";
				return false;
			}
		}
		return true; // every clause satisfied
	}


	bool ClauseDB::simplify(const Vass& vass, unsigned model_size)
	{
		//assert( va.check_counter() );

		// If we have the empty clause, DB is inconsistent
		if (has_empty()) return false;

		// cerr << "Inside simplify(), va.size(): " << va.size() << "\n";
		// Simplify from all unit clauses (by unit propagation)
		Vass va = vass;
		const auto lit2cl = make_lit2cl();
		if (unit_propagate(0,va,nullptr,lit2cl) > 0) {
			for (auto cl = db.begin(); cl != db.end(); ) {
				bool removed = false;
				auto i = cl->begin();
				for ( ; i != cl->end(); ) {
					if (va.get_var(*i).val == *i) {
						// Remove clause
						cl = db.erase(cl);
						removed = true;
						break;
					} else if (va.get_var(*i).val == -(*i)) {
						// Remove literal
						i = cl->erase(i);
						// do NOT assume clause is non-tautology => no break statement
					} else {
						++i;
					}
				}
				if (!removed) ++cl;
			}
			return true;
		} else {
			return false;
		}
	}

	int ClauseDB::dpll(
		Vass& vass, 
		unsigned model_size, 
		DPLL_Selection& select,
		deadline_t deadline,
		const std::vector<pvar>* suggestions)
	{
		vass.reset(model_size);

		if (has_empty()) return 0;

		// Fill variable assignment list with unit literals and modes
		/* It is allowed for Vass to specify additional assignments 
			and whether they should be open/closed branches
			Max ones/assignment restrictions are however taken directly from va,
			setting those in vass has no effect
		*/
		//// TODO: REMOVE ===>
		//db.clear();
		//PClause cl; cl.insert(1); cl.insert(31); cl.insert(-2); this->insert(move(cl));
		//cl.clear(); cl.insert(1); cl.insert(-3); this->insert(move(cl));
		//cl.clear(); cl.insert(2); cl.insert(3); cl.insert(4); this->insert(move(cl));
		//cl.clear(); cl.insert(-4); cl.insert(-5); this->insert(move(cl));
		//cl.clear(); cl.insert(21); cl.insert(-4); cl.insert(-6); this->insert(move(cl));
		//cl.clear(); cl.insert(5); cl.insert(6); this->insert(move(cl));
		////PClause cl; cl.insert(1); cl.insert(2); this->insert(move(cl));
		////cl.clear(); cl.insert(1); cl.insert(3); cl.insert(7); this->insert(move(cl));
		////cl.clear(); cl.insert(-2); cl.insert(-3); cl.insert(4); this->insert(move(cl));
		////cl.clear(); cl.insert(-4); cl.insert(5); cl.insert(8); this->insert(move(cl));
		////cl.clear(); cl.insert(-4); cl.insert(6); cl.insert(9); this->insert(move(cl));
		////cl.clear(); cl.insert(-5); cl.insert(-6); this->insert(move(cl));

		//vass.set_max1(1000);
		//vass.init_table(100);
		//// <====== END OF REMOVE

		// Build table literal -> clauses for fast unit propagation
		auto lit2cl = make_lit2cl();

		// Copy all unconditional assignments
		for (int a : ass) {
			const int stat = vass.insert(pvar(a,-1,true));
			if (stat > 0) {
				// Propagate
				//std::cerr << "Init, propagating: " << a << "\n";
				const int ustat = unit_propagate(-1,vass,nullptr,lit2cl,deadline);
				if (ustat <= 0) {
					return ustat;
				}
			} else {
				// insert failed => fail
				return stat;
			}
		}
		//std::cerr << "DPLL2, m1: " << vass.max1() << "\n";

		// Add suggestions
		if (suggestions) {
			for (const pvar& s : *suggestions) {
				pvar& pv = vass.get_var(s.val);
				if (!pv.is_set()) {
					DEBUG_WARNING(std::cerr << "Warning: ignoring DPLL nil suggestion " << pv.val << "*" << pv.both_ways << "\n");
				} else if (pv.val == -s.val) {
					return 0; // contradicts propagated unit literals
				} else {
					pvar tmp; // use default constructor to construct all arguments
					tmp.level = -1;
					tmp.sublevel = 0;
					tmp.ante = nullptr;
					// Suggestions: value and closed/open
					tmp.val = s.val;
					tmp.both_ways = s.both_ways;
					pv = tmp;
				}
			}
		}

		DEBUG_TRACE( cerr << "DPLL, initial assignments: " << vass << "\n" );
		assert( vass.check_counter() );

		// Settings for clause learning
		const int param_learn_max_occurs = params.is_set(parameters::dpll_cll_max_occurs);
		const int param_learn_clause_length = params.is_set(parameters::dpll_cll_clause_length);

		int level = -1;	
		try {
			for (;;) {
				if (vass.size() == model_size) break; // successfully found model
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				// Make decision: pick first unassigned literal with VSIDS
				const int clit = select.choose(*this,vass);
				//// TODO: DELETE+====>
				//static int arr[] = { -21, -31, -1 };
				////static int arr[] = { -7, -8, -9, -1 };
				//static int arr_idx = 0;
				//const int clit = arr[arr_idx++];
				//// <======= END
				DEBUG_TRACE(cerr << "Decision: " << clit << "\n");
				assert(clit != 0);
				assert( vass.get_var(clit).val == 0 );
				level = 1 + vass.size(); // assignment levels starts at 1
				const int s = vass.insert(pvar(clit,level,0));
				if (s <= 0) {
					std::cerr << "ERROR: DPLL selection strategy wanted to assign invalid: " << clit << "\n";
					exit(1);
				}

				for (;;) {
					// Unit Propagate
					std::pair<const PClause*,int> err(nullptr,0);
					auto* eptr = &err;
					const int pstat = unit_propagate(level,vass,&err,lit2cl,deadline);
					if (pstat <= 0) {
						// Propagation failed, backtrack
						switch (pstat) {
						case -1:
							if (param_learn_max_occurs) eptr = nullptr; // no clause learning for max_occur limit reached
							break;
						case -2:
							if (param_learn_clause_length) eptr = nullptr; // no clause learning for clause length reached
							break;
						}
						const int bstat = backtrack(level,vass,select,eptr);
						if (bstat <= 0) return bstat; // fail
						//continue; // go back to propagation
					} else {
						// Propagation successful
						break;
					}
				}
			}
		} catch (max_assignments) {
			// max permitted number of assignments reached
			//DEBUG_INFO(cout << "warning: maximal " << va.acounter() << " dpll assignments reached\n");
			throw;
		} catch (unmodifiable) {
			return 0; // fail
		}
		// In debug mode, check that we have a correct model
		assert( this->satisfied(vass) );
		return 1;
	}

	std::multimap<int,PClause*> ClauseDB::make_lit2cl()
	{
		std::multimap<int,PClause*> lit2cl;
		for (PClause& cl : db) {
			for (int l : cl) {
				lit2cl.insert(std::make_pair(l,&cl));
			}
		}
		return lit2cl;
	}

	int ClauseDB::unit_propagate(
		int level, 
		Vass& vass, 
		std::pair<const PClause*,int>* err, 
		const std::multimap<int,PClause*>& lit2cl,
		deadline_t deadline)
	{
		int sublevel = 0;
		// Resolve last assignment
		for (Vass::size_type vi = vass.size() - 1; vi < vass.size() && !vass.empty(); ++vi) {
			if (std::chrono::steady_clock::now() > deadline) throw time_out();
			int flit = -vass[vi].val;
			//cerr << "Resolving: " << flit << "\n";
			// Check if any clauses are affected
			auto rng = lit2cl.equal_range(flit);
			for (auto i = rng.first; i != rng.second; ++i) {
				PClause& cl = *(i->second);
				const int lit2 = cl.other_watched(flit);
				if (lit2 == 0) { continue; } // flit not watched => clause is not unit
				////cerr << "Checking prop against: " << cl << " (" << i << " of " << db.size() << ")\n";
				// Check if there is a third literal that is NOT set to false
				int lit3 = 0;
				for (auto l = cl.begin(); l != cl.end(); ++l) {
					if (*l == flit || *l == lit2) continue;
					pvar& v = vass.get_var(*l);
					if (v.val != 0) {
						if (v.val == *l) {
							// found it: set to true, therefore not false (optimization)
							lit3 = *l;
							break;
						} else if (v.val == -*l) {
							// set to false => not set to true (this assumes we never have tautologies)
							// Note: for now we don't do this optimization
						}
					} else {
						// unset => not set to false => watch it
						lit3 = *l; 
						break;
					}
				}

				if (lit3 == 0) {
					// No third literal, this clause is unit now
					// Check if assignment of lit2 is consistent
					//cerr << " trying to insert unit " << lit2 << " into vass...\n";
					const int ins = vass.insert(pvar(lit2,level,++sublevel,&cl));
					if (ins <= 0) {
						if (err) {
							err->first = &cl;
						}
						return ins;
					}
					//cerr << "success\n";
				} else {
					// Third literal, this clause is still not unit
					// Update watch from lit to lit3
					cl.switch_watched(flit,lit3);
				}
			} // for each clause: check if assignment propagates unit literal
		} // while there are more propagations
		return 1;
	}

	int ClauseDB::dual_horn_solve(Vass& vass, unsigned model_size)
	{
		std::cerr << "dual_horn, modelsize: " << model_size << "\n";
		vass.reset(model_size);
#ifndef NDEBUG
		// Check that all clauses are dual horn (= have max 1 negative literal)
		auto is_dual_horn = [](const PClause& cl) -> bool {
			int negs = 0;
			for (int l : cl) {
				if (l < 0 && ++negs > 1) return false;
			}
			return true;
		};
		for (const auto& cl : db) {
			if (!is_dual_horn(cl)) {
				std::cerr << "ERROR! Not Dual Horn clause: " << cl << "\n";
				assert(false);
				exit(1);
			}
		}
#endif
		auto lit2cl = make_lit2cl();
		const auto res = unit_propagate(0,vass,nullptr,lit2cl);
		if (res <= 0) return res; // failed
		std::cerr << "Dual horn, after UP: " << vass << "\n";
		// Set remaining literals to true
		for (int k = 1; k <= int(model_size); ++k) {
			pvar& pv = vass.get_var(k);
			if (!pv.is_set()) {
				std::cerr << "Setting: " << pv.val << " -> " << k << "\n";
				const auto r = vass.insert(pvar(k,0,0,false));
				assert(r > 0);
				std::cerr << "vass: " << vass << "\n";
			} else {
				std::cerr << "Already set: " << k << " = " << pv.val << "\n";
			}
		}
		std::cerr << "Dual horn, after all: " << vass << "\n";
		return 1;
	}


	//int ClauseDB::unit_propagate(int level, Vass& vass, DPLL_Selection& select, std::pair<const PClause*,int>* err)
	//{
	//	int sublevel = 0;
	//	// Propagate falsehood from assignment
	//	for (Vass::size_type vi = vass.size() - 1; vi < vass.size() && !vass.empty(); ++vi) {
	//		int flit = -vass[vi].val;
	//		//cerr << "Propagating: " << flit << "\n";
	//		// Check if any clauses are affected
	//		for (PClause& cl : db) {
	//			const int lit2 = cl.other_watched(flit);
	//			if (lit2 == 0) { continue; } // flit not watched => clause is not unit
	//			////cerr << "Checking prop against: " << cl << " (" << i << " of " << db.size() << ")\n";
	//			// Check if there is a third literal that is NOT set to false
	//			int lit3 = 0;
	//			for (auto l = cl.begin(); l != cl.end(); ++l) {
	//				if (*l == flit || *l == lit2) continue;
	//				pvar& v = vass.get_var(*l);
	//				if (v.val != 0) {
	//					if (v.val == *l) {
	//						// found it: set to true, therefore not false (optimization)
	//						lit3 = *l;
	//						break;
	//					} else if (v.val == -*l) {
	//						// set to false => not set to true (this assumes we never have tautologies)
	//						// Note: for now we don't do this optimization
	//					}
	//				} else {
	//					// unset => not set to false => watch it
	//					lit3 = *l; 
	//					break;
	//				}
	//			}

	//			if (lit3 == 0) {
	//				// No third literal, this clause is unit now
	//				// Check if assignment of lit2 is consistent
	//				//cerr << " trying to insert unit " << lit2 << " into vass...\n";
	//				const int ins = vass.insert(pvar(lit2,level,++sublevel,&cl));
	//				if (ins <= 0) {
	//					if (err) {
	//						err->first = &cl;
	//					}
	//					return ins;
	//				}
	//				//cerr << "success\n";
	//			} else {
	//				// Third literal, this clause is still not unit
	//				// Update watch from lit to lit3
	//				cl.switch_watched(flit,lit3);
	//			}
	//		} // for each clause: check if assignment propagates unit literal
	//	} // while there are more propagations
	//	return 1;
	//}


	int ClauseDB::backtrack(int level, Vass& vass, DPLL_Selection& select, std::pair<const PClause*,int>* err)
	{
		// Clause Learning?
		const int cllmax = params.force_int(parameters::dpll_cll);
		const int cllsize = params.force_int(parameters::dpll_cllsize);
		// TODO: more elegant/user-friendly representation of learning schemes, support for more?
		// 0 = no clause learning, 1 = 1UIP, 2 = rel_sat, 3 = Decision (=all)
		const int cl_learn = params.force_int(parameters::dpll_learn);

		int uip = 0; // have uip?
		if (err && err->first && cl_learn > 0 && cllsize > 0 && int(db.size()) < cllmax) {
			DEBUG_TRACE(cerr << "inconsistent: " << vass << "\n");
			// Inconsistent, learn clause
			// lit2 propagated by cl
			// -lit2 can be found from lit2->ante
			PClause ccl;

			if (cl_learn >= 3) {
				// Not 1UIP => learn full clause
				for (const pvar* p : vass) {
					if (p->is_decision()) {
						assert(p->val != 0);
						ccl.insert(-p->val);
					}
				}
			} else {
				// Use 1UIP or rel_sat
				const bool use_uip = (cl_learn == 1);
				ccl = *err->first;
				// Note: when using UIP with max_one violations, one literal might be 0
				for (auto i = ccl.begin(); i != ccl.end(); ) {
					const pvar& pv = vass.get_var(*i);
					if (!pv.is_set()) {
						i = ccl.erase(i);
					} else ++i;
				}

				// Note: to ensure we find UIP, we need to resolve backwards according to vass
				// TODO: remove k
				int k = 0;
				for ( ; k < 100000; ++k) {
					//std::cerr << "  updated constraint clause: " << ccl << "\n";
					// Get LATEST implied literal, find UIP
					pvar* imp_lit = nullptr;
					int sublevel = -1;
					int count = 0;
					for (int i : ccl) {
						pvar& var = vass.get_var(i);
						assert(var.val != 0 || !(std::cerr << "i = " << i << " <-- " << ccl << "\n"));
						if (var.level == level) {
							++count;
							if (var.ante && var.sublevel > sublevel) {
								imp_lit = &var;
								sublevel = var.sublevel;
							}
						}
					}
					if (count == 1 && use_uip) {
						// this is the first UIP
						uip = 1;
						break;
					}
					if (!imp_lit) break; // nothing more to resolve

					// Resolve ccl with imp_lit->ante, resolving literal is imp_lit->val
					//std::cerr << "Resolving " << ccl << " with " << *imp_lit->ante << "\n";
					assert(ccl.has(imp_lit->val) || ccl.has(-imp_lit->val));
					PClause ccl2 = ccl;
					int rl = imp_lit->val;
					if (!ccl2.erase(rl)) {
						rl = -rl;
						ccl2.erase(rl);
					}
					assert(imp_lit->ante->has(-rl));
					for (int i : *imp_lit->ante) {
						if (i != -rl) ccl2.insert(i);
					}
					if (ccl2 == ccl) break; // no change
					ccl = std::move(ccl2);
				}

				if (k >= 10000) {
					std::cerr << "ERROR: resolving, ccl: " << ccl << "\n";
					exit(1);
				}
			}


			// Add constraint to DB and update counter (clause learning)
			assert(!ccl.empty());
			DEBUG_TRACE( cerr << "Adding constraint: " << ccl << "\n" );
			DEBUG_TRACE( cin.sync(); string ss; getline(cin,ss); );
			if (!ccl.unit()) {
				// Add only if clause size is smaller or equal to err->second
				if (int(ccl.size()) <= cllsize) {
					select.update(*this,vass,ccl);
					insert(std::move(ccl));
				}
			} // else: already flipped
		}

		// Backtrack until we may flip
		while (!vass.empty() && !vass.flip_back()) vass.pop();
		if (vass.empty()) return 0; // unsat

		return 1;
	}


	void ClauseDB::print(ostream& os) const
	{
		os << "Assignments:\n";
		for_each(ass.begin(),ass.end(),[&](int a){
			os << " " << a << "\n";
		});
		os << "Clauses:\n";
		for_each(db.begin(),db.end(),[&](const PClause& c){
			os << " " << c << "\n";
		});
		if (has_empty()) {
			os << "KB inconsistent due to presence of empty clause\n";
		}
	}



}



